package djmongo

import (
	"djconstants"
	"djextrafunc"
	"djlogger"
	"encoding/json"
	mgo "mongo"
	"mongo/bson"
	openrtb3 "openrtb/openrtb3"
	"strconv"
	"strings"
	"time"
	"fmt"
	"math"

	openrtb "openrtb/openrtb2.5"
)

// SerializeRequestData Type
type SerializeRequestData struct {
	Id        bson.ObjectId              `bson:"_id" json:"_id,omitempty"`
	RequestId string                     `bson:"requestId" json:"requestId,omitempty"`
	Data      string                     `bson:"data" json:"data"`
	BidderID  int                        `bson:"bidderid" json:"bidderid"`
	Cur       string                     `bson:"cur" json:"cur"`
	Geo       *openrtb3.Geo              `bson:"geo" json:"geo"`
	Device    *openrtb3.Device           `bson:"device" json:"device"`
	Site      *openrtb3.Site             `bson:"site" json:"site"`
	User      *openrtb3.User             `bson:"user" json:"user"`
	Deal      map[string][]openrtb3.Deal `bson:"deal" json:"deal"`
	PMP       map[string]int8            `bson:"pmp" json:"pmp"`
	Datetime  string                     `bson:"datetime" json:"datetime"`
}

// SerializeRequestData Type 2.5
type SerializeRequestData25 struct {
	Id        bson.ObjectId             `bson:"_id" json:"_id,omitempty"`
	RequestId string                    `bson:"requestId" json:"requestId,omitempty"`
	Data      string                    `bson:"data" json:"data"`
	BidderID  int                       `bson:"bidderid" json:"bidderid"`
	Cur       string                    `bson:"cur" json:"cur"`
	Geo       *openrtb.Geo              `bson:"geo" json:"geo"`
	Device    *openrtb.Device           `bson:"device" json:"device"`
	Site      *openrtb.Site             `bson:"site" json:"site"`
	User      *openrtb.User             `bson:"user" json:"user"`
	Deal      map[string][]openrtb.Deal `bson:"deal" json:"deal"`
	PMP       map[string]int8           `bson:"pmp" json:"pmp"`
	Datetime  string                    `bson:"datetime" json:"datetime"`
}

// SerializeResponseData Type
type SerializeResponseData struct {
	Id                 bson.ObjectId              `bson:"_id" json:"_id,omitempty"`
	ResponseId         string                     `bson:"responseId" json:"responseId,omitempty"`
	Data               string                     `bson:"data" json:"data"`
	RequestID          string                     `bson:"requestid" json:"requestid"`
	BidderID           int                        `bson:"bidderid" json:"bidderid"`
	Cur                string                     `bson:"cur" json:"cur"`
	Geo                *openrtb3.Geo              `bson:"geo" json:"geo"`
	Device             *openrtb3.Device           `bson:"device" json:"device"`
	Site               *openrtb3.Site             `bson:"site" json:"site"`
	User               *openrtb3.User             `bson:"user" json:"user"`
	Deal               map[string][]openrtb3.Deal `bson:"deal" json:"deal"`
	PMP                map[string]int8            `bson:"pmp" json:"pmp"`
	Win_notice         int                        `bson:"win_notice" json:"win_notice"`
	Loss_notice        int                        `bson:"loss_notice" json:"loss_notice"`
	Testing_Win_notice int                        `bson:"test_win_notice" json:"test_win_notice"`
	Adminshare         float64                    `bson:"adminshare" json:"adminshare"`
	Price              float64                    `bson:"price" json:"price"`
	Click              int                        `bson:"click" json:"click"`
	Impression         int                        `bson:"impression" json:"impression"`
	Datetime           string                     `bson:"datetime" json:"datetime"`
	Ad_id int                       `bson:"ad_id" json:"ad_id"`

}

// SerializeResponseData Type 2.5
type SerializeResponseData25 struct {
	Id                 bson.ObjectId             `bson:"_id" json:"_id,omitempty"`
	ResponseId         string                    `bson:"responseId" json:"responseId,omitempty"`
	Data               string                    `bson:"data" json:"data"`
	RequestID          string                    `bson:"requestid" json:"requestid"`
	BidderID           int                       `bson:"bidderid" json:"bidderid"`
	Cur                string                    `bson:"cur" json:"cur"`
	Geo                *openrtb.Geo              `bson:"geo" json:"geo"`
	Device             *openrtb.Device           `bson:"device" json:"device"`
	Site               *openrtb.Site             `bson:"site" json:"site"`
	User               *openrtb.User             `bson:"user" json:"user"`
	Deal               map[string][]openrtb.Deal `bson:"deal" json:"deal"`
	PMP                map[string]int8           `bson:"pmp" json:"pmp"`
	Win_notice         int                       `bson:"win_notice" json:"win_notice"`
	Loss_notice        int                       `bson:"loss_notice" json:"loss_notice"`
	Testing_Win_notice int                       `bson:"test_win_notice" json:"test_win_notice"`
	Ad_id int                       `bson:"ad_id" json:"ad_id"`
	Adminshare         float64                   `bson:"adminshare" json:"adminshare"`
	Price              float64                   `bson:"price" json:"price"`
	Click              int                       `bson:"click" json:"click"`
	Impression         int                       `bson:"impression" json:"impression"`
	Datetime           string                    `bson:"datetime" json:"datetime"`
}

// SerializeWinLossNoticeData Type
type SerializeWinLossNoticeData struct {
	RequestId string  `bson:"requestId" json:"requestId,omitempty"`
	URL       string  `bson:"url" json:"url"`
	Bidid     string  `bson:"bidid" json:"bidid"`
	AuctionId string  `bson:"auctionId" json:"auctionId"`
	Price     float64 `bson:"price" json:"price"`
	Currency  string  `bson:"currency" json:"currency"`
	Impid     string  `bson:"impid" json:"impid"`
	Seatid    string  `bson:"seatid" json:"seatid"`
	Adid      string  `bson:"adid" json:"adid"`
	Lossid    string  `bson:"lossid,omitempty" json:"lossid,omitempty"`
	Screening int32   `bson:"screening" json:"screening"`
	Datetime  string  `bson:"datetime" json:"datetime"`
}

// Mongo Connection Function
func MongoConnect() (*mgo.Session, error) {
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured : ", err, " Recovered from panic")
		}
	}()
	info := &mgo.DialInfo{
		Addrs:    []string{djconstants.MongoHost},
		Timeout:  1 * time.Second,
		Database: djconstants.MongoDbName,
		Username: djconstants.MongoUser,
		Password: djconstants.MongoPass,
	}
	Session, err := mgo.DialWithInfo(info)
	if err != nil {
		djlogger.Log.Println("Error in connecting to MongoDB! ", err)
	}
	return Session, err
}

// Mongo Collection Function
func MongoCollection(collectionName string) (*mgo.Session, *mgo.Collection, error) {
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured : ", err, " Recovered from panic")
		}
	}()
	client, err := MongoConnect()
	if err != nil {
		return nil, nil, err
	}
	collection := client.DB(djconstants.MongoDbName).C(collectionName)
	return client, collection, nil
}

// Save Request Function
func MongoSaveRequest(dsp_id int, req *openrtb3.Request) (string, bson.ObjectId, SerializeRequestData, bool) {
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured : ", err, " Recovered from panic")
		}
	}()
	var reqData SerializeRequestData
	t := time.Now()
	y := t.Year()
	mon := int(t.Month())
	d := t.Day()
	h := t.Hour()
	deal := make(map[string][]openrtb3.Deal)
	pmp := make(map[string]int8)
	json, e := json.Marshal(req)
	min := GetMin(t.Minute())
	tableSuffix := strconv.Itoa(y) + strconv.Itoa(mon) + strconv.Itoa(d) + strconv.Itoa(h) + strconv.Itoa(min)
	cur := djextrafunc.ArrayToString(req.Cur, ",")
	if e != nil {
		return "", "", reqData, false
	}
	for k, item := range req.Item {
		deal["deal"+strconv.Itoa(k)] = item.Deal
		pmp["pmp"+strconv.Itoa(k)] = item.Private
	}
	// Get a handle for your collection
	client, collection, err := MongoCollection(djconstants.MongoRequestTable + "_" + tableSuffix)
	if err != nil {
		return "", "", reqData, false
	}
	defer client.Close()
	// Insert a single document
	objid := bson.NewObjectId()
	objidRes := bson.NewObjectId()
	responseId := "30" + objidRes.Hex() + "_" + tableSuffix
	reqData = SerializeRequestData{Id: objid, RequestId: req.ID, Data: string(json), BidderID: dsp_id, Cur: cur, Geo: req.Context.Device.Geo, Device: req.Context.Device, Site: req.Context.Site, User: req.Context.User, Deal: deal, PMP: pmp, Datetime: time.Now().Format("2006-01-02 15:04:05")}
	err = collection.Insert(reqData)
	if err != nil {
		reqData = SerializeRequestData{}
		return "", "", reqData, false
	}
	djlogger.Log.Println("Request Inserted")
	return responseId, objidRes, reqData, true
}

// Save Request Function 2.5
func MongoSaveRequest25(dsp_id int, req *openrtb.BidRequest) (string, bson.ObjectId, SerializeRequestData25, bool) {
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured : ", err, " Recovered from panic")
		}
	}()
	var reqData SerializeRequestData25
	t := time.Now()
	y := t.Year()
	mon := int(t.Month())
	d := t.Day()
	h := t.Hour()
	deal := make(map[string][]openrtb.Deal)
	pmp := make(map[string]int8)
	json, e := json.Marshal(req)
	min := GetMin(t.Minute())
	tableSuffix := strconv.Itoa(y) + strconv.Itoa(mon) + strconv.Itoa(d) + strconv.Itoa(h) + strconv.Itoa(min)
	cur := djextrafunc.ArrayToString(req.Cur, ",")
	if e != nil {
		return "", "", reqData, false
	}
	for k, imp := range req.Imp {
		if imp.Pmp != nil {
			deal["deal"+strconv.Itoa(k)] = imp.Pmp.Deals
			pmp["pmp"+strconv.Itoa(k)] = imp.Pmp.Private
		}
	}
	// Get a handle for your collection
	client, collection, err := MongoCollection(djconstants.MongoRequestTable + "_" + tableSuffix)
	if err != nil {
		return "", "", reqData, false
	}
	defer client.Close()
	// Insert a single document
	objid := bson.NewObjectId()
	objidRes := bson.NewObjectId()
	responseId := "25" + objidRes.Hex() + "_" + tableSuffix
	reqData = SerializeRequestData25{Id: objid, RequestId: req.ID, Data: string(json), BidderID: dsp_id, Cur: cur, Geo: req.Device.Geo, Device: req.Device, Site: req.Site, User: req.User, Deal: deal, PMP: pmp, Datetime: time.Now().Format("2006-01-02 15:04:05")}
	err = collection.Insert(reqData)
	if err != nil {
		reqData = SerializeRequestData25{}
		return "", "", reqData, false
	}
	djlogger.Log.Println("Request Inserted")
	return responseId, objidRes, reqData, true
}

// Save Response Function
func MongoSaveResponse(responseId string, objid bson.ObjectId, dsp_id int,ad_id int, reqData SerializeRequestData, output map[string]interface{}) bool {
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured : ", err, " Recovered from panic")
		}
	}()
	t := time.Now()
	y := t.Year()
	mon := int(t.Month())
	d := t.Day()
	h := t.Hour()
	min := GetMin(t.Minute())
	tableSuffix := strconv.Itoa(y) + strconv.Itoa(mon) + strconv.Itoa(d) + strconv.Itoa(h) + strconv.Itoa(min)
	json, e := json.Marshal(output)
	if e != nil {
		djlogger.Log.Println("Response Map can't able to convert into json")
	}
	// Get a handle for your collection
	client, collection, err := MongoCollection(djconstants.MongoResponseTable + "_" + tableSuffix)
	if err != nil {
		return false
	}
	defer client.Close()
	// Insert document
	err = collection.Insert(SerializeResponseData{Id: objid, Ad_id: ad_id, ResponseId: responseId, Data: string(json), RequestID: reqData.RequestId, BidderID: dsp_id, Cur: reqData.Cur, Geo: reqData.Geo, Device: reqData.Device, Site: reqData.Site, User: reqData.User, Deal: reqData.Deal, PMP: reqData.PMP, Win_notice: 0, Loss_notice: 0, Testing_Win_notice: 0, Adminshare: 0.0, Click: 0, Impression: 0, Datetime: time.Now().Format("2006-01-02 15:04:05")})
	if err != nil {
		return false
	}
	djlogger.Log.Println("Response Inserted")
	return true
}

// Save Response Function
func MongoSaveResponse25(responseId string, objid bson.ObjectId, dsp_id int,ad_id int, reqData SerializeRequestData25, output map[string]interface{}) bool {
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured : ", err, " Recovered from panic")
		}
	}()
	t := time.Now()
	y := t.Year()
	mon := int(t.Month())
	d := t.Day()
	h := t.Hour()
	min := GetMin(t.Minute())
	tableSuffix := strconv.Itoa(y) + strconv.Itoa(mon) + strconv.Itoa(d) + strconv.Itoa(h) + strconv.Itoa(min)
	json, e := json.Marshal(output)
	if e != nil {
		djlogger.Log.Println("Response Map can't able to convert into json")
	}
	// Get a handle for your collection
	client, collection, err := MongoCollection(djconstants.MongoResponseTable + "_" + tableSuffix)
	if err != nil {
		return false
	}
	defer client.Close()
	// Insert document
	err = collection.Insert(SerializeResponseData25{Id: objid,Ad_id: ad_id,  ResponseId: responseId, Data: string(json), RequestID: reqData.RequestId, BidderID: dsp_id, Cur: reqData.Cur, Geo: reqData.Geo, Device: reqData.Device, Site: reqData.Site, User: reqData.User, Deal: reqData.Deal, PMP: reqData.PMP, Win_notice: 0, Loss_notice: 0, Testing_Win_notice: 0, Adminshare: 0.0, Click: 0, Impression: 0, Datetime: time.Now().Format("2006-01-02 15:04:05")})
	if err != nil {
		return false
	}
	djlogger.Log.Println("Response Inserted")
	return true
}

// Save Win or Loss Notice Function
func MongoSaveWinLossNotice(ProcessDetails map[string]interface{}, NoticeType int) bool {
	djlogger.Log.Println("MongoSaveWinLossNotice called")
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured mongo: ", err, " Recovered from panic")
		}
	}()
	
	
	var reqPrice string
	switch v := ProcessDetails["price"].(type) {
	case []byte:
		reqPrice = string(v)
	default:
		reqPrice = v.(string)
	}
	var (
		screening     int32
		auction_price float64
		resultRequest string
		resultBI      int
		resultWN      int
		resultLN      int
		resultTWN     int
		resultAS      float64
		admin_share   float64
		resultPr      float64
	)
	t := time.Now()
	y := t.Year()
	mon := int(t.Month())
	d := t.Day()
	h := t.Hour()
	min := GetMin(t.Minute())
	tableSuffix := strconv.Itoa(y) + strconv.Itoa(mon) + strconv.Itoa(d) + strconv.Itoa(h) + strconv.Itoa(min)
	version, bidid := trimString(ProcessDetails["bidid"].(string))
	restableSuffix := strings.Split(bidid, "_")
	oid := bson.ObjectIdHex(restableSuffix[0])
	// Get a handle for your collection
	clientRes, collectionRes, err := MongoCollection(djconstants.MongoResponseTable + "_" + restableSuffix[1])
	if err != nil {
		djlogger.Log.Println("Fetching mongo collectionn error")
		return false
	}
	
	
	defer clientRes.Close()
	if NoticeType == 1 {
		// Query One
		if version == "30" {
			result := SerializeResponseData{}
			err := collectionRes.Find(bson.M{"_id": oid}).One(&result)
			if err != nil {
				djlogger.Log.Println("Error : Cannot find ID in response collection table")
				return false
			}
			resultBI = result.BidderID
			resultRequest = result.RequestID
			resultWN = result.Win_notice
			resultTWN = result.Testing_Win_notice
			resultAS = result.Adminshare
			resultPr = result.Price
		} else {
			result := SerializeResponseData25{}
			err := collectionRes.Find(bson.M{"_id": oid}).One(&result)
			if err != nil {
				djlogger.Log.Println("Error : Cannot find ID in response collection table")
				return false
			}
			resultBI = result.BidderID
			resultRequest = result.RequestID
			resultWN = result.Win_notice
			resultTWN = result.Testing_Win_notice
			resultAS = result.Adminshare
			resultPr = result.Price
		}

		if reqPrice == "SCREENING" {
			auction_price = 0.0
			screening = 1
			collectionRes.Update(bson.M{"_id": oid}, bson.M{"$set": bson.M{"test_win_notice": resultTWN + 1}})
		} else {
			fmt.Println("394")
			auction_price = djextrafunc.StringToFloat(reqPrice, 64)
					

			screening = 0
			_, _, admin_share, _, _, _ = djextrafunc.GetDspId("", "", resultBI, nil)
			admin_share = ((auction_price / 1000)) *( admin_share /100)
			admin_share = roundFloat(admin_share, 6)
			cpm:= auction_price/1000
			cpm = roundFloat(cpm, 6)


			collectionRes.Update(bson.M{"_id": oid}, bson.M{"$set": bson.M{"win_notice": resultWN + 1}})
			collectionRes.Update(bson.M{"_id": oid}, bson.M{"$set": bson.M{"price": resultPr + cpm}})
			collectionRes.Update(bson.M{"_id": oid}, bson.M{"$set": bson.M{"adminshare": resultAS + admin_share}})
		}
		
	
		// Get a handle for your collection
		client, collection, err := MongoCollection(djconstants.MongoWinNoticeTable + "_" + tableSuffix)
		if err != nil {
			djlogger.Log.Println("Error :", err)
			return false
		}
		defer client.Close()
		// Insert document
		cpm:= auction_price/1000
		cpm = roundFloat(cpm, 6)
		err = collection.Insert(SerializeWinLossNoticeData{RequestId: resultRequest, URL: ProcessDetails["url"].(string), Bidid: bidid, AuctionId: ProcessDetails["auctionId"].(string), Price: cpm, Currency: ProcessDetails["currency"].(string), Impid: ProcessDetails["impid"].(string), Seatid: ProcessDetails["seatid"].(string), Adid: ProcessDetails["adid"].(string), Screening: screening, Datetime: time.Now().Format("2006-01-02 15:04:05")})
		if err != nil {
			djlogger.Log.Println("Error :", err)
			return false
		}
		djlogger.Log.Println("Response Updated")
		djlogger.Log.Println("Win Notice Inserted")
	} else {
		// Query One
		if version == "30" {
			result := SerializeResponseData{}
			err := collectionRes.Find(bson.M{"_id": oid}).One(&result)
			if err != nil {
				djlogger.Log.Println("Error : Cannot find ID in response collection table")
				return false
			}
			resultLN = result.Loss_notice
		} else {
			result := SerializeResponseData25{}
			err := collectionRes.Find(bson.M{"_id": oid}).One(&result)
			if err != nil {
				djlogger.Log.Println("Error : Cannot find ID in response collection table")
				return false
			}
			resultLN = result.Loss_notice
		}
		// Update Loss Notice in response collection
		collectionRes.Update(bson.M{"_id": oid}, bson.M{"$set": bson.M{"loss_notice": resultLN + 1}})
		djlogger.Log.Println("Response Updated")
		if reqPrice == "SCREENING" {
			auction_price = 0.0
		} else {
			auction_price = djextrafunc.StringToFloat(reqPrice, 64)
			auction_price /= 1000
		}
		// Get a handle for your collection
		client, collection, err := MongoCollection(djconstants.MongoLossNoticeTable + "_" + tableSuffix)
		if err != nil {
			djlogger.Log.Println("Error :", err)
			return false
		}
		defer client.Close()
		// Insert document
		cpm:= auction_price/1000
		cpm = roundFloat(cpm, 6)
		err = collection.Insert(SerializeWinLossNoticeData{URL: ProcessDetails["url"].(string), Bidid: bidid, AuctionId: ProcessDetails["auctionId"].(string), Price: cpm, Currency: ProcessDetails["currency"].(string), Impid: ProcessDetails["impid"].(string), Seatid: ProcessDetails["seatid"].(string), Adid: ProcessDetails["adid"].(string), Lossid: ProcessDetails["lossid"].(string), Datetime: time.Now().Format("2006-01-02 15:04:05")})
		if err != nil {
			djlogger.Log.Println("Error :", err)
			return false
		}
		djlogger.Log.Println("Loss Notice Inserted")
	}
	return true
}

// Save Win or Loss Notice Function
func MongoSaveClickImpression(ProcessDetails map[string]interface{}, Type int) bool {
	defer func() {
		if err := recover(); err != nil {
			djlogger.Log.Println("Error occured : ", err, " Recovered from panic")
		}
	}()
	var (
		resultC int
		resultI int
	)
	version, bidid := trimString(ProcessDetails["bidid"].(string))
	restableSuffix := strings.Split(bidid, "_")
	oid := bson.ObjectIdHex(restableSuffix[0])
	// Get a handle for your collection
	clientRes, collectionRes, err := MongoCollection(djconstants.MongoResponseTable + "_" + restableSuffix[1])
	if err != nil {
		return false
	}
	defer clientRes.Close()
	// Query One
	if version == "30" {
		result := SerializeResponseData{}
		err := collectionRes.Find(bson.M{"_id": oid}).One(&result)
		if err != nil {
			djlogger.Log.Println("Cannot find ID in response collection table")
			return false
		}
		resultC = result.Click
		resultI = result.Impression
	} else {
		result := SerializeResponseData25{}
		err := collectionRes.Find(bson.M{"_id": oid}).One(&result)
		if err != nil {
			djlogger.Log.Println("Cannot find ID in response collection table")
			return false
		}
		resultC = result.Click
		resultI = result.Impression
	}
	if Type == 1 {
		collectionRes.Update(bson.M{"_id": oid}, bson.M{"$set": bson.M{"click": resultC + 1}})
	} else {
		collectionRes.Update(bson.M{"_id": oid}, bson.M{"$set": bson.M{"impression": resultI + 1}})
	}
	return true
}

// Function for getting minutes
func GetMin(k int) (m int) {
	if k <= 15 && k >= 0 {
		m = 0
	} else if k <= 30 && k > 15 {
		m = 15
	} else if k <= 45 && k > 30 {
		m = 30
	} else if k <= 59 && k > 45 {
		m = 45
	}
	return
}

// Function for triming first 2 characters
func trimString(s string) (string, string) {
	return s[:2], s[2:]
}

func roundFloat(val float64, precision uint) float64 {
	ratio := math.Pow(10, float64(precision))
	return math.Round(val*ratio) / ratio
}